'use strict';

const { Transform } = require('stream');

const _FTYP = Buffer.from([0x66, 0x74, 0x79, 0x70]); // ftyp
const _MOOV = Buffer.from([0x6d, 0x6f, 0x6f, 0x76]); // moov
const _MDHD = Buffer.from([0x6d, 0x64, 0x68, 0x64]); // mdhd
const _MOOF = Buffer.from([0x6d, 0x6f, 0x6f, 0x66]); // moof
const _MDAT = Buffer.from([0x6d, 0x64, 0x61, 0x74]); // mdat
const _TFHD = Buffer.from([0x74, 0x66, 0x68, 0x64]); // tfhd
const _TRUN = Buffer.from([0x74, 0x72, 0x75, 0x6e]); // trun
const _MFRA = Buffer.from([0x6d, 0x66, 0x72, 0x61]); // mfra
const _HVCC = Buffer.from([0x68, 0x76, 0x63, 0x43]); // hvcC
const _HEV1 = Buffer.from([0x68, 0x65, 0x76, 0x31]); // hev1
const _HVC1 = Buffer.from([0x68, 0x76, 0x63, 0x31]); // hvc1
const _AVCC = Buffer.from([0x61, 0x76, 0x63, 0x43]); // avcC
const _AVC1 = Buffer.from([0x61, 0x76, 0x63, 0x31]); // avc1
const _AVC2 = Buffer.from([0x61, 0x76, 0x63, 0x32]); // avc2
const _AVC3 = Buffer.from([0x61, 0x76, 0x63, 0x33]); // avc3
const _AVC4 = Buffer.from([0x61, 0x76, 0x63, 0x34]); // avc4
const _MP4A = Buffer.from([0x6d, 0x70, 0x34, 0x61]); // mp4a
const _ESDS = Buffer.from([0x65, 0x73, 0x64, 0x73]); // esds
const _HLS_INIT_DEF = true; // hls playlist available after initialization and before 1st segment
const _HLS_SIZE_DEF = 4; // hls playlist size default
const _HLS_SIZE_MIN = 2; // hls playlist size minimum
const _HLS_SIZE_MAX = 20; // hls playlist size maximum
const _HLS_EXTRA_DEF = 0; // hls playlist extra segments in memory default
const _HLS_EXTRA_MIN = 0; // hls playlist extra segments in memory minimum
const _HLS_EXTRA_MAX = 10; // hls playlist extra segments in memory maximum
const _SEG_SIZE_DEF = 2; // segment list size default
const _SEG_SIZE_MIN = 2; // segment list size minimum
const _SEG_SIZE_MAX = 30; // segment list size maximum
const _MOOF_SEARCH_LIMIT = 50; // number of allowed attempts to find missing moof atom

/**
 * @file
 * <ul>
 * <li>Creates a stream transform for piping a fmp4 (fragmented mp4) from ffmpeg.</li>
 * <li>Can be used to generate a fmp4 m3u8 HLS playlist and compatible file fragments.</li>
 * <li>Can be used for storing past segments of the mp4 video in a buffer for later access.</li>
 * <li>Must use the following ffmpeg args <b><i>-movflags +frag_keyframe+empty_moov+default_base_moof</i></b> to generate
 * a valid fmp4 with a compatible file structure : ftyp+moov -> moof+mdat -> moof+mdat -> moof+mdat ...</li>
 * </ul>
 * @extends stream.Transform
 */
class Mp4Frag extends Transform {
  /**
   * @constructor
   * @param {object} [options] - Configuration options.
   * @param {boolean} [options.readableObjectMode = false] - If true, segments will be piped out as an object instead of a Buffer.
   * @param {string} [options.hlsPlaylistBase] - Base name of files in m3u8 playlist. Affects the generated m3u8 playlist by naming file fragments. Must be set to generate m3u8 playlist. e.g. 'front_door'
   * @param {number} [options.hlsPlaylistSize = 4] - Number of segments to use in m3u8 playlist. Must be an integer ranging from 2 to 20.
   * @param {number} [options.hlsPlaylistExtra = 0] - Number of extra segments to keep in memory. Must be an integer ranging from 0 to 10.
   * @param {boolean} [options.hlsPlaylistInit = true] - Indicates that m3u8 playlist should be generated after [initialization]{@link Mp4Frag#initialization} is created and before media segments are created.
   * @param {number} [options.segmentCount = 2] - Number of segments to keep in memory. Has no effect if using options.hlsPlaylistBase. Must be an integer ranging from 2 to 30.
   * @throws Will throw an error if options.hlsPlaylistBase contains characters other than letters(a-zA-Z) and underscores(_).
   */
  constructor(options) {
    options = options instanceof Object ? options : {};
    super({ writableObjectMode: false, readableObjectMode: options.readableObjectMode === true });
    if (typeof options.hlsPlaylistBase !== 'undefined') {
      if (/[^a-z_]/gi.test(options.hlsPlaylistBase)) {
        throw new Error('hlsPlaylistBase must only contain underscores and case-insensitive letters (_, a-z, A-Z)');
      }
      this._hlsPlaylistBase = options.hlsPlaylistBase;
      this._hlsPlaylistInit = Mp4Frag._validateBoolean(options.hlsPlaylistInit, _HLS_INIT_DEF);
      this._hlsPlaylistSize = Mp4Frag._validateNumber(options.hlsPlaylistSize, _HLS_SIZE_DEF, _HLS_SIZE_MIN, _HLS_SIZE_MAX);
      this._hlsPlaylistExtra = Mp4Frag._validateNumber(options.hlsPlaylistExtra, _HLS_EXTRA_DEF, _HLS_EXTRA_MIN, _HLS_EXTRA_MAX);
      this._segmentCount = this._hlsPlaylistSize + this._hlsPlaylistExtra;
      this._segmentObjects = [];
    } else if (typeof options.segmentCount !== 'undefined') {
      this._segmentCount = Mp4Frag._validateNumber(options.segmentCount, _SEG_SIZE_DEF, _SEG_SIZE_MIN, _SEG_SIZE_MAX);
      this._segmentObjects = [];
    }
    if (options.readableObjectMode === true) {
      this._sendSegment = this._sendSegmentBufferObject;
    } else {
      this._sendSegment = this._sendSegmentBuffer;
    }
    this._parseChunk = this._findFtyp;
    return this;
  }

  /**
   * @readonly
   * @property {?string} audioCodec
   * - Returns the audio codec information as a <b>string</b>.
   * <br/>
   * - Returns <b>null</b> if requested before [initialized event]{@link Mp4Frag#event:initialized}.
   * @returns {?string}
   */
  get audioCodec() {
    return this._audioCodec || null;
  }

  /**
   * @readonly
   * @property {?string} videoCodec
   * - Returns the video codec information as a <b>string</b>.
   * <br/>
   * - Returns <b>null</b> if requested before [initialized event]{@link Mp4Frag#event:initialized}.
   * @returns {?string}
   */
  get videoCodec() {
    return this._videoCodec || null;
  }

  /**
   * @readonly
   * @property {?string} mime
   * - Returns the mime type information as a <b>string</b>.
   * <br/>
   * - Returns <b>null</b> if requested before [initialized event]{@link Mp4Frag#event:initialized}.
   * @returns {?string}
   */
  get mime() {
    return this._mime || null;
  }

  /**
   * @readonly
   * @property {?Buffer} initialization
   * - Returns the Mp4 initialization fragment as a <b>Buffer</b>.
   * <br/>
   * - Returns <b>null</b> if requested before [initialized event]{@link Mp4Frag#event:initialized}.
   * @returns {?Buffer}
   */
  get initialization() {
    return this._initialization || null;
  }

  /**
   * @readonly
   * @property {?Buffer} segment
   * - Returns the latest Mp4 segment as a <b>Buffer</b>.
   * <br/>
   * - Returns <b>null</b> if requested before first [segment event]{@link Mp4Frag#event:segment}.
   * @returns {?Buffer}
   */
  get segment() {
    return this._segment || null;
  }

  /**
   * @readonly
   * @property {object} segmentObject
   * - Returns the latest Mp4 segment as an <b>object</b>.
   * <br/>
   *  - <b><code>{segment, sequence, duration, timestamp, keyframe}</code></b>
   * <br/>
   * - Returns <b>{segment: null, sequence: -1, duration: -1; timestamp: -1, keyframe: true}</b> if requested before first [segment event]{@link Mp4Frag#event:segment}.
   * @returns {object}
   */
  get segmentObject() {
    return {
      segment: this.segment,
      sequence: this.sequence,
      duration: this.duration,
      timestamp: this.timestamp,
      keyframe: this.keyframe,
    };
  }

  /**
   * @readonly
   * @property {number} timestamp
   * - Returns the timestamp of the latest Mp4 segment as an <b>Integer</b>(<i>milliseconds</i>).
   * <br/>
   * - Returns <b>-1</b> if requested before first [segment event]{@link Mp4Frag#event:segment}.
   * @returns {number}
   */
  get timestamp() {
    return this._timestamp || -1;
  }

  /**
   * @readonly
   * @property {number} duration
   * - Returns the duration of latest Mp4 segment as a <b>Float</b>(<i>seconds</i>).
   * <br/>
   * - Returns <b>-1</b> if requested before first [segment event]{@link Mp4Frag#event:segment}.
   * @returns {number}
   */
  get duration() {
    return this._duration || -1;
  }

  /**
   * @readonly
   * @property {number} totalDuration
   * - Returns the total duration of all Mp4 segments as a <b>Float</b>(<i>seconds</i>).
   * <br/>
   * - Returns <b>-1</b> if requested before first [segment event]{@link Mp4Frag#event:segment}.
   * @returns {number}
   */
  get totalDuration() {
    return this._totalDuration || -1;
  }

  /**
   * @readonly
   * @property {number} totalByteLength
   * - Returns the total byte length of the Mp4 initialization and all Mp4 segments as ant <b>Integer</b>(<i>bytes</i>).
   * <br/>
   * - Returns <b>-1</b> if requested before [initialized event]{@link Mp4Frag#event:initialized}.
   * @returns {number}
   */
  get totalByteLength() {
    return this._totalByteLength || -1;
  }

  /**
   * @readonly
   * @property {?string} m3u8
   * - Returns the fmp4 HLS m3u8 playlist as a <b>string</b>.
   * <br/>
   * - Returns <b>null</b> if requested before [initialized event]{@link Mp4Frag#event:initialized}.
   * @returns {?string}
   */
  get m3u8() {
    return this._m3u8 || null;
  }

  /**
   * @readonly
   * @property {number} sequence
   * - Returns the sequence of the latest Mp4 segment as an <b>Integer</b>.
   * <br/>
   * - Returns <b>-1</b> if requested before first [segment event]{@link Mp4Frag#event:segment}.
   * @returns {number}
   */
  get sequence() {
    return Number.isInteger(this._sequence) ? this._sequence : -1;
  }

  /**
   * @readonly
   * @property {boolean} keyframe
   * - Returns a boolean indicating if the current segment contains a keyframe.
   * <br/>
   * - Returns <b>false</b> if the current segment does not contain a keyframe.
   * <br/>
   * - Returns <b>true</b> if segment only contains audio.
   * @returns {boolean}
   */
  get keyframe() {
    return typeof this._keyframe === 'boolean' ? this._keyframe : true;
  }

  /**
   * @readonly
   * @property {boolean} allKeyframes
   * - Returns a boolean indicating if all segments contain a keyframe.
   * <br/>
   * - Returns <b>false</b> if any segments do not contain a keyframe.
   * @returns {boolean}
   */
  get allKeyframes() {
    return typeof this._allKeyframes === 'boolean' ? this._allKeyframes : true;
  }

  /**
   * @readonly
   * @property {?Array} segmentObjects
   * - Returns the Mp4 segments as an <b>Array</b> of <b>objects</b>
   * <br/>
   * - <b><code>[{segment, sequence, duration, timestamp, keyframe},...]</code></b>
   * <br/>
   * - Returns <b>null</b> if requested before first [segment event]{@link Mp4Frag#event:segment}.
   * @returns {?Array}
   */
  get segmentObjects() {
    return this._segmentObjects && this._segmentObjects.length ? this._segmentObjects : null;
  }

  /**
   * @param {number|string} sequence - sequence number
   * @returns {?object}
   * - Returns the Mp4 segment that corresponds to the numbered sequence as an <b>object</b>.
   * <br/>
   * - <b><code>{segment, sequence, duration, timestamp, keyframe}</code></b>
   * <br/>
   * - Returns <b>null</b> if there is no segment that corresponds to sequence number.
   */
  getSegmentObject(sequence) {
    sequence = Number.parseInt(sequence);
    if (this._segmentObjects && this._segmentObjects.length) {
      return this._segmentObjects[this._segmentObjects.length - 1 - (this._sequence - sequence)] || null;
    }
    return null;
  }

  /**
   * Clear cached values
   */
  resetCache() {
    /**
     * Fires when resetCache() is called.
     * @event Mp4Frag#reset
     * @type {Event}
     */
    this.emit('reset');
    this._parseChunk = this._findFtyp;
    if (this._segmentObjects) {
      this._segmentObjects = [];
    }
    this._timescale = undefined;
    this._sequence = undefined;
    this._allKeyframes = undefined;
    this._keyframe = undefined;
    this._mime = undefined;
    this._videoCodec = undefined;
    this._audioCodec = undefined;
    this._initialization = undefined;
    this._segment = undefined;
    this._timestamp = undefined;
    this._duration = undefined;
    this._totalDuration = undefined;
    this._totalByteLength = undefined;
    this._moof = undefined;
    this._mdatBuffer = undefined;
    this._moofLength = undefined;
    this._mdatLength = undefined;
    this._mdatBufferSize = undefined;
    this._ftyp = undefined;
    this._ftypLength = undefined;
    this._m3u8 = undefined;
    this._setKeyFrame = undefined;
  }

  /**
   * Search buffer for ftyp.
   * @param {Buffer} chunk
   * @private
   */
  _findFtyp(chunk) {
    const chunkLength = chunk.length;
    if (chunkLength < 8 || chunk.indexOf(_FTYP) !== 4) {
      this.emit('error', new Error(`${_FTYP.toString()} not found.`));
      return;
    }
    this._ftypLength = chunk.readUInt32BE(0);
    if (this._ftypLength < chunkLength) {
      this._ftyp = chunk.slice(0, this._ftypLength);
      this._parseChunk = this._findMoov;
      this._parseChunk(chunk.slice(this._ftypLength));
    } else if (this._ftypLength === chunkLength) {
      this._ftyp = chunk;
      this._parseChunk = this._findMoov;
    } else {
      //should not be possible to get here because ftyp is approximately 24 bytes
      //will have to buffer this chunk and wait for rest of it on next pass
      this.emit('error', new Error(`ftypLength:${this._ftypLength} > chunkLength:${chunkLength}`));
      //return;
    }
  }

  /**
   * Search buffer for moov.
   * @param {Buffer} chunk
   * @private
   */
  _findMoov(chunk) {
    const chunkLength = chunk.length;
    if (chunkLength < 8 || chunk.indexOf(_MOOV) !== 4) {
      this.emit('error', new Error(`${_MOOV.toString()} not found.`));
      return;
    }
    const moovLength = chunk.readUInt32BE(0);
    if (moovLength < chunkLength) {
      this._initialize(Buffer.concat([this._ftyp, chunk], this._ftypLength + moovLength));
      this._ftyp = undefined;
      this._ftypLength = undefined;
      this._parseChunk = this._findMoof;
      this._parseChunk(chunk.slice(moovLength));
    } else if (moovLength === chunkLength) {
      this._initialize(Buffer.concat([this._ftyp, chunk], this._ftypLength + moovLength));
      this._ftyp = undefined;
      this._ftypLength = undefined;
      this._parseChunk = this._findMoof;
    } else {
      //probably should not arrive here here because moov is typically < 800 bytes
      //will have to store chunk until size is big enough to have entire moov piece
      //ffmpeg may have crashed before it could output moov and got us here
      this.emit('error', new Error(`moovLength:${moovLength} > chunkLength:${chunkLength}`));
      //return;
    }
  }

  /**
   * Parse moov for mime.
   * @fires Mp4Frag#initialized
   * @param {Buffer} chunk
   * @private
   */
  _initialize(chunk) {
    this._initialization = chunk;
    const mdhdIndex = chunk.indexOf(_MDHD);
    const mdhdVersion = chunk[mdhdIndex + 4];
    this._timescale = chunk.readUInt32BE(mdhdIndex + (mdhdVersion === 0 ? 16 : 24));
    this._timestamp = Date.now();
    this._sequence = -1;
    this._allKeyframes = true;
    this._totalDuration = 0;
    this._totalByteLength = chunk.byteLength;
    this._setKeyFrame = () => {};
    const codecs = [];
    let mp4Type;
    if (this._parseCodecAVCC(chunk) || this._parseCodecHVCC(chunk)) {
      codecs.push(this._videoCodec);
      mp4Type = 'video';
    }
    if (this._parseCodecMP4A(chunk)) {
      codecs.push(this._audioCodec);
      if (!this._videoCodec) {
        mp4Type = 'audio';
      }
    }
    if (codecs.length === 0) {
      this.emit('error', new Error(`codecs not found.`));
      return;
    }
    this._mime = `${mp4Type}/mp4; codecs="${codecs.join(', ')}"`;
    if (this._hlsPlaylistBase && this._hlsPlaylistInit) {
      let m3u8 = '#EXTM3U\n';
      m3u8 += '#EXT-X-VERSION:7\n';
      m3u8 += `#EXT-X-TARGETDURATION:1\n`;
      m3u8 += `#EXT-X-MEDIA-SEQUENCE:0\n`;
      m3u8 += `#EXT-X-MAP:URI="init-${this._hlsPlaylistBase}.mp4"\n`;
      this._m3u8 = m3u8;
    }
    /**
     * Fires when the [initialization]{@link Mp4Frag#initialization} of the Mp4 is parsed from the piped data.
     * @event Mp4Frag#initialized
     * @type {Event}
     * @property {object} object
     * @property {string} object.mime - [Mp4Frag.mime]{@link Mp4Frag#mime}
     * @property {Buffer} object.initialization - [Mp4Frag.initialization]{@link Mp4Frag#initialization}
     * @property {string} object.m3u8 - [Mp4Frag.m3u8]{@link Mp4Frag#m3u8}
     */
    this.emit('initialized', { mime: this.mime, initialization: this.initialization, m3u8: this.m3u8 });
  }

  /**
   * Find moof after miss due to corrupt data in pipe.
   * @param {Buffer} chunk
   * @private
   */
  _moofSearch(chunk) {
    if (this._moofSearches < _MOOF_SEARCH_LIMIT) {
      this._moofSearches++;
      //console.warn(`MOOF search attempt number ${this._moofSearches}.`);
      const index = chunk.indexOf(_MOOF);
      if (index > 3 && chunk.length > index + 3) {
        this._moofSearches = undefined;
        this._parseChunk = this._findMoof;
        this._parseChunk(chunk.slice(index - 4));
      }
    } else {
      this.emit('error', new Error(`${_MOOF.toString()} search failed after ${this._moofSearches} attempts.`));
      //return;
    }
  }

  /**
   * Search buffer for moof.
   * @param {Buffer} chunk
   * @private
   */
  _findMoof(chunk) {
    if (this._moofBuffer) {
      this._moofBuffer.push(chunk);
      const chunkLength = chunk.length;
      this._moofBufferSize += chunkLength;
      if (this._moofLength === this._moofBufferSize) {
        //todo verify this works
        this._moof = Buffer.concat(this._moofBuffer, this._moofLength);
        this._moofBuffer = undefined;
        this._moofBufferSize = undefined;
        this._parseChunk = this._findMdat;
      } else if (this._moofLength < this._moofBufferSize) {
        this._moof = Buffer.concat(this._moofBuffer, this._moofLength);
        const sliceIndex = chunkLength - (this._moofBufferSize - this._moofLength);
        this._moofBuffer = undefined;
        this._moofBufferSize = undefined;
        this._parseChunk = this._findMdat;
        this._parseChunk(chunk.slice(sliceIndex));
      }
    } else {
      const chunkLength = chunk.length;
      if (chunkLength < 8 || chunk.indexOf(_MOOF) !== 4) {
        // ffmpeg occasionally pipes corrupt data, lets try to get back to normal if we can find next MOOF box before attempts run out
        const mfraIndex = chunk.indexOf(_MFRA);
        if (mfraIndex !== -1) {
          // console.log(`MFRA was found at ${mfraIndex}. This is expected at the end of stream.`);
          return;
        }
        // console.warn('Failed to find MOOF. Starting MOOF search. Ignore this if your file stream input has ended.');
        this._moofSearches = 0;
        this._parseChunk = this._moofSearch;
        this._parseChunk(chunk);
        return;
      }
      this._moofLength = chunk.readUInt32BE(0);
      if (this._moofLength === 0) {
        this.emit('error', new Error(`Bad data from input stream reports ${_MOOF.toString()} length of 0.`));
        return;
      }
      if (this._moofLength < chunkLength) {
        this._moof = chunk.slice(0, this._moofLength);
        this._parseChunk = this._findMdat;
        this._parseChunk(chunk.slice(this._moofLength));
      } else if (this._moofLength === chunkLength) {
        // todo verify this works
        this._moof = chunk;
        this._parseChunk = this._findMdat;
      } else {
        this._moofBuffer = [chunk];
        this._moofBufferSize = chunkLength;
      }
    }
  }

  /**
   * Search buffer for mdat.
   * @param {Buffer} chunk
   * @private
   */
  _findMdat(chunk) {
    if (this._mdatBuffer) {
      this._mdatBuffer.push(chunk);
      const chunkLength = chunk.length;
      this._mdatBufferSize += chunkLength;
      if (this._mdatLength === this._mdatBufferSize) {
        this._setSegment(Buffer.concat([this._moof, ...this._mdatBuffer], this._moofLength + this._mdatLength));
        this._moof = undefined;
        this._mdatBuffer = undefined;
        this._mdatBufferSize = undefined;
        this._mdatLength = undefined;
        this._moofLength = undefined;
        this._parseChunk = this._findMoof;
      } else if (this._mdatLength < this._mdatBufferSize) {
        this._setSegment(Buffer.concat([this._moof, ...this._mdatBuffer], this._moofLength + this._mdatLength));
        const sliceIndex = chunkLength - (this._mdatBufferSize - this._mdatLength);
        this._moof = undefined;
        this._mdatBuffer = undefined;
        this._mdatBufferSize = undefined;
        this._mdatLength = undefined;
        this._moofLength = undefined;
        this._parseChunk = this._findMoof;
        this._parseChunk(chunk.slice(sliceIndex));
      }
    } else {
      const chunkLength = chunk.length;
      if (chunkLength < 8 || chunk.indexOf(_MDAT) !== 4) {
        this.emit('error', new Error(`${_MDAT.toString()} not found.`));
        return;
      }
      this._mdatLength = chunk.readUInt32BE(0);
      if (this._mdatLength > chunkLength) {
        this._mdatBuffer = [chunk];
        this._mdatBufferSize = chunkLength;
      } else if (this._mdatLength === chunkLength) {
        this._setSegment(Buffer.concat([this._moof, chunk], this._moofLength + chunkLength));
        this._moof = undefined;
        this._moofLength = undefined;
        this._mdatLength = undefined;
        this._parseChunk = this._findMoof;
      } else {
        this._setSegment(Buffer.concat([this._moof, chunk], this._moofLength + this._mdatLength));
        const sliceIndex = this._mdatLength;
        this._moof = undefined;
        this._moofLength = undefined;
        this._mdatLength = undefined;
        this._parseChunk = this._findMoof;
        this._parseChunk(chunk.slice(sliceIndex));
      }
    }
  }

  /**
   * Set hvcC keyframe.
   * @param {Buffer} chunk
   * @private
   */
  _setKeyFrameHVCC(chunk) {
    let index = this._moofLength + 8;
    const end = chunk.length - 5;
    while (index < end) {
      const nalLength = chunk.readUInt32BE(index);
      // simplify check for iframe nal types 16, 17, 18, 19, 20, 21; (chunk[(index += 4)] & 0x20) >> 1
      if ((chunk[(index += 4)] & 0x20) === 32) {
        this._keyframe = true;
        return;
      }
      index += nalLength;
    }
    this._allKeyframes = false;
    this._keyframe = false;
  }

  /**
   * Set avcC keyframe.
   * @see {@link https://github.com/video-dev/hls.js/blob/729a36d409cc78cc391b17a0680eaf743f9213fb/tools/mp4-inspect.js#L48}
   * @param {Buffer} chunk
   * @private
   */
  _setKeyFrameAVCC(chunk) {
    let index = this._moofLength + 8;
    const end = chunk.length - 5;
    while (index < end) {
      const nalLength = chunk.readUInt32BE(index);
      if ((chunk[(index += 4)] & 0x1f) === 5) {
        this._keyframe = true;
        return;
      }
      index += nalLength;
    }
    this._allKeyframes = false;
    this._keyframe = false;
  }

  /**
   * Set duration and timestamp.
   * @see {@link https://github.com/video-dev/hls.js/blob/04cc5f167dac2aed4e41e493125968838cb32445/src/utils/mp4-tools.ts#L392}
   * @param {Buffer} chunk
   * @private
   */
  _setDurTime(chunk) {
    const duration = (() => {
      const trunIndex = chunk.indexOf(_TRUN);
      let trunOffset = trunIndex + 4;
      const trunFlags = chunk.readUInt32BE(trunOffset);
      trunOffset += 4;
      const sampleCount = chunk.readUInt32BE(trunOffset);
      // prefer using trun sample durations
      if (trunFlags & 0x000100) {
        trunOffset += 4;
        trunFlags & 0x000001 && (trunOffset += 4);
        trunFlags & 0x000004 && (trunOffset += 4);
        const increment = 4 + (trunFlags & 0x000200 && 4) + (trunFlags & 0x000400 && 4) + (trunFlags & 0x000800 && 4);
        let sampleDurationSum = 0;
        for (let i = 0; i < sampleCount; ++i, trunOffset += increment) {
          sampleDurationSum += chunk.readUInt32BE(trunOffset);
        }
        return sampleDurationSum / this._timescale;
      }
      // fallback to using tfhd default sample duration
      const tfhdIndex = chunk.indexOf(_TFHD);
      let tfhdOffset = tfhdIndex + 4;
      const tfhdFlags = chunk.readUInt32BE(tfhdOffset);
      if (tfhdFlags & 0x000008) {
        tfhdOffset += 8;
        tfhdFlags & 0x000001 && (tfhdOffset += 8);
        tfhdFlags & 0x000002 && (tfhdOffset += 4);
        return (chunk.readUInt32BE(tfhdOffset) * sampleCount) / this._timescale;
      }
      return 0;
    })();
    const currentTime = Date.now();
    const elapsed = (currentTime - this._timestamp) / 1000;
    this._timestamp = currentTime;
    this._duration = duration || elapsed;
  }

  /**
   * Process current segment.
   * @fires Mp4Frag#segment
   * @param {Buffer} chunk
   * @private
   */
  _setSegment(chunk) {
    this._segment = chunk;
    this._setKeyFrame(chunk);
    this._setDurTime(chunk);
    this._sequence++;
    if (this._segmentObjects) {
      this._segmentObjects.push({
        segment: chunk,
        sequence: this._sequence,
        duration: this._duration,
        timestamp: this._timestamp,
        keyframe: this._keyframe,
      });
      this._totalDuration += this._duration;
      this._totalByteLength += chunk.byteLength;
      while (this._segmentObjects.length > this._segmentCount) {
        const {
          duration,
          segment: { byteLength },
        } = this._segmentObjects.shift();
        this._totalDuration -= duration;
        this._totalByteLength -= byteLength;
      }
      if (this._hlsPlaylistBase) {
        let i = this._segmentObjects.length > this._hlsPlaylistSize ? this._segmentObjects.length - this._hlsPlaylistSize : 0;
        const mediaSequence = this._segmentObjects[i].sequence;
        let targetDuration = 1;
        let segments = '';
        for (i; i < this._segmentObjects.length; ++i) {
          targetDuration = Math.max(targetDuration, this._segmentObjects[i].duration);
          segments += `#EXTINF:${this._segmentObjects[i].duration.toFixed(6)},\n`;
          segments += `${this._hlsPlaylistBase}${this._segmentObjects[i].sequence}.m4s\n`;
        }
        let m3u8 = '#EXTM3U\n';
        m3u8 += '#EXT-X-VERSION:7\n';
        m3u8 += `#EXT-X-TARGETDURATION:${Math.round(targetDuration) || 1}\n`;
        m3u8 += `#EXT-X-MEDIA-SEQUENCE:${mediaSequence}\n`;
        m3u8 += `#EXT-X-MAP:URI="init-${this._hlsPlaylistBase}.mp4"\n`;
        m3u8 += segments;
        this._m3u8 = m3u8;
      }
    } else {
      this._totalDuration = this._duration;
      this._totalByteLength = this._initialization.byteLength + chunk.byteLength;
    }
    this._sendSegment();
    /**
     * Fires when the latest Mp4 segment is parsed from the piped data.
     * @event Mp4Frag#segment
     * @type {Event}
     * @property {object} object - [Mp4Frag.segmentObject]{@link Mp4Frag#segmentObject}
     * @property {Buffer} object.segment - [Mp4Frag.segment]{@link Mp4Frag#segment}
     * @property {number} object.sequence - [Mp4Frag.sequence]{@link Mp4Frag#sequence}
     * @property {number} object.duration - [Mp4Frag.duration]{@link Mp4Frag#duration}
     * @property {number} object.timestamp - [Mp4Frag.timestamp]{@link Mp4Frag#timestamp}
     * @property {number} object.keyframe - [Mp4Frag.keyframe]{@link Mp4Frag#keyframe}
     */
    this.emit('segment', this.segmentObject);
  }

  /**
   * @private
   */
  _sendSegmentBuffer() {
    this.emit('data', this.segment, { type: 'segment', sequence: this.sequence, duration: this.duration, timestamp: this.timestamp, keyframe: this.keyframe });
  }

  /**
   * @private
   */
  _sendSegmentBufferObject() {
    this.emit('data', { type: 'segment', segment: this.segment, sequence: this.sequence, duration: this.duration, timestamp: this.timestamp, keyframe: this.keyframe });
  }

  /**
   * @param {Buffer} chunk
   * @returns {boolean}
   * @private
   */
  _parseCodecMP4A(chunk) {
    const index = chunk.indexOf(_MP4A);
    if (index !== -1) {
      const codec = ['mp4a'];
      const esdsIndex = chunk.indexOf(_ESDS, index);
      // verify tags 3, 4, 5 to be in expected positions
      if (esdsIndex !== -1 && chunk[esdsIndex + 8] === 0x03 && chunk[esdsIndex + 16] === 0x04 && chunk[esdsIndex + 34] === 0x05) {
        codec.push(chunk[esdsIndex + 21].toString(16));
        codec.push(((chunk[esdsIndex + 39] & 0xf8) >> 3).toString());
        this._audioCodec = codec.join('.');
        return true;
      }
      // console.warn('unexpected mp4a esds structure');
    }
    return false;
  }

  /**
   * @param {Buffer} chunk
   * @returns {boolean}
   * @private
   */
  _parseCodecAVCC(chunk) {
    const index = chunk.indexOf(_AVCC);
    if (index !== -1) {
      const codec = [];
      if (chunk.includes(_AVC1)) {
        codec.push('avc1');
      } else if (chunk.includes(_AVC2)) {
        codec.push('avc2');
      } else if (chunk.includes(_AVC3)) {
        codec.push('avc3');
      } else if (chunk.includes(_AVC4)) {
        codec.push('avc4');
      } else {
        return false;
      }
      codec.push(
        chunk
          .slice(index + 5, index + 8)
          .toString('hex')
          .toUpperCase()
      );
      this._videoCodec = codec.join('.');
      this._setKeyFrame = this._setKeyFrameAVCC;
      return true;
    }
    return false;
  }

  /**
   * @param {Buffer} chunk
   * @returns {boolean}
   * @private
   */
  _parseCodecHVCC(chunk) {
    const index = chunk.indexOf(_HVCC);
    if (index !== -1) {
      const codec = [];
      if (chunk.includes(_HVC1)) {
        codec.push('hvc1');
      } else if (chunk.includes(_HEV1)) {
        codec.push('hev1');
      } else {
        return false;
      }
      const tmpByte = chunk[index + 5];
      const generalProfileSpace = tmpByte >> 6; // get 1st 2 bits (11000000)
      const generalTierFlag = !!(tmpByte & 0x20) ? 'H' : 'L'; // get next bit (00100000)
      const generalProfileIdc = (tmpByte & 0x1f).toString(); // get last 5 bits (00011111)
      const generalProfileCompatibility = Mp4Frag._reverseBitsToHex(chunk.readUInt32BE(index + 6));
      const generalConstraintIndicator = Buffer.from(chunk.slice(index + 10, index + 16).filter(byte => !!byte)).toString('hex');
      const generalLevelIdc = chunk[index + 16].toString();
      switch (generalProfileSpace) {
        case 0:
          codec.push(generalProfileIdc);
          break;
        case 1:
          codec.push(`A${generalProfileIdc}`);
          break;
        case 2:
          codec.push(`B${generalProfileIdc}`);
          break;
        case 3:
          codec.push(`C${generalProfileIdc}`);
          break;
      }
      codec.push(generalProfileCompatibility);
      codec.push(`${generalTierFlag}${generalLevelIdc}`);
      if (generalConstraintIndicator.length) {
        codec.push(generalConstraintIndicator);
      }
      this._videoCodec = codec.join('.');
      this._setKeyFrame = this._setKeyFrameHVCC;
      return true;
    }
    return false;
  }

  /**
   * Required for stream transform.
   * @param {Buffer} chunk
   * @param {string} encoding
   * @param {TransformCallback} callback
   * @private
   */
  _transform(chunk, encoding, callback) {
    this._parseChunk(chunk);
    callback();
  }

  /**
   * Run cleanup when unpiped.
   * @param {TransformCallback} callback
   * @private
   */
  _flush(callback) {
    this.resetCache();
    callback();
  }

  /**
   * Validate number is in range.
   * @param {*} n
   * @param {number} def
   * @param {number} min
   * @param {number} max
   * @returns {number}
   * @private
   * @static
   */
  static _validateNumber(n, def, min, max) {
    n = Number.parseInt(n);
    return isNaN(n) ? def : n < min ? min : n > max ? max : n;
  }

  /**
   * Validate boolean value.
   * @param {*} bool
   * @param {boolean} def
   * @returns {boolean}
   * @private
   * @static
   */
  static _validateBoolean(bool, def) {
    return typeof bool === 'boolean' ? bool : def;
  }

  /**
   * Reverse bits and convert to hexadecimal.
   * @see {@link http://graphics.stanford.edu/~seander/bithacks.html#ReverseParallel}
   * @param {number} n - unsigned 32 bit integer
   * @returns {string} - bit reversed hex string
   * @private
   * @static
   */
  static _reverseBitsToHex(n) {
    n = ((n >> 1) & 0x55555555) | ((n & 0x55555555) << 1);
    n = ((n >> 2) & 0x33333333) | ((n & 0x33333333) << 2);
    n = ((n >> 4) & 0x0f0f0f0f) | ((n & 0x0f0f0f0f) << 4);
    n = ((n >> 8) & 0x00ff00ff) | ((n & 0x00ff00ff) << 8);
    return ((n >> 16) | (n << 16)).toString(16);
  }
}

module.exports = Mp4Frag;
